---
title: Widget configurations · Cloudflare Turnstile docs
description: Configure your Turnstile widget's appearance, behavior, and
  functionality using data attributes or JavaScript render parameters.
lastUpdated: 2025-09-04T15:24:34.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/turnstile/get-started/client-side-rendering/widget-configurations/
  md: https://developers.cloudflare.com/turnstile/get-started/client-side-rendering/widget-configurations/index.md
---

Configure your Turnstile widget's appearance, behavior, and functionality using data attributes or JavaScript render parameters.

## Rendering methods

Turnstile widgets can be implemented using implicit or explicit rendering.

* Implicit rendering

  Implicit rendering automatically scans your HTML for elements with the `cf-turnstile` class and renders the widget when the page loads. It is best used for simple implementations, static websites, or when you want widgets to appear immediately on page load.

  **How it works**

  1. Add the Turnstile script to your page.
  2. Include `<div class="cf-turnstile" data-sitekey="your-key"></div>` elements.
  3. Widgets will render automatically when the page loads.
  4. Configure the widget using `data-*` attributes on the HTML element.

  ```html
    <script src="https://challenges.cloudflare.com/turnstile/v0/api.js" async defer></script>
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-theme="light"></div>
  ```

* Explicit rendering

  Explicit rendering gives you programmatic control over when and how widgets are created using JavaScript functions. It is best used for dynamic websites and single-page applications (SPAs), when you need to control timing of widget creation, conditional rendering based on visitor interactions, or for multiple widgets with different configurations.

  **How it works**

  1. Add the Turnstile script with `?render=explicit` parameter.
  2. Create container elements (without the `cf-turnstile` class).
  3. Call `turnstile.render()` function when you want to create widgets.
  4. Configure the widget using JavaScript object parameters.

  ```html
    <script src="https://challenges.cloudflare.com/turnstile/v0/api.js?render=explicit" defer></script>
    <div id="my-widget"></div>


    <script>
    window.onload = function() {
      turnstile.render('#my-widget', {
        sitekey: '<YOUR-SITE-KEY>',
        theme: 'light',
        callback: function(token) {
          console.log('Success:', token);
        }
      });
    };
    </script>
  ```

***

## Widget sizes

The Turnstile widget can have two different fixed sizes or a flexible width size when using the Managed or Non-Interactive modes.

| Size | Width | Height | Use case |
| - | - | - | - |
| Normal | 300px | 65px | Standard implementation |
| Flexible | 100% (min: 300px) | 65px | Responsive design |
| Compact | 150px | 140px | Space-constrained layouts |

* `normal`: The default size works well for most desktop and mobile layouts. Use this if you have adequate horizontal space on your website or form.
* `flexible`: Automatically adapts to the container width while maintaining minimum usability. Use this for responsive designs that need to work across all screen sizes.
* `compact`: Ideal for mobile interfaces, sidebars, or any space where horizontal space is limited. The compact widget is taller than normal to accommodate the smaller width.

Note

Widget size only applies to Managed and Non-Interactive modes. Invisible widgets have no visual footprint regardless of size configuration.

* Implicit rendering

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>"></div>
  ```

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-size="flexible"></div>
  ```

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-size="compact"></div>
  ```

* Explicit rendering

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>'
    });
  ```

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>',
      size: 'flexible'
    });
  ```

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>',
      size: 'compact'
    });
  ```

***

## Theme options

Customize the widget's visual appearance to match your website's design.

* `auto` (default): Automatically matches the visitor's system theme preference. Auto is recommended for most implementations as it respects the visitor's preferences and provides the best accessibility experience.
* `light`: Light theme with bright colors and clear contrast. Light theme works best on bright backgrounds and provides high contrast for readability.
* `dark`: Dark theme optimized for dark interfaces. Dark theme is ideal for dark interfaces, gaming sites, or applications with dark color schemes.

- Implicit rendering

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>"></div>
  ```

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-theme="light"></div>
  ```

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-theme="dark"></div>
  ```

- Explicit rendering

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>'
    });
  ```

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>',
      theme: 'light'
    });
  ```

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>',
      theme: 'dark'
    });
  ```

***

## Appearance modes

Control when the widget becomes visible to visitors using the appearance mode.

* `always` (default): The widget is always visible from page load. This is the best option for most implementations where you want your visitors to see the widget immediately as it provides clear visual feedback that security verification is in place.
* `execute`: The widget only becomes visible after the challenge begins. This is useful for when you need to control the timing of widget appearance, such as showing it only when a visitor starts filling out a form or selecting a submit button.
* `interaction-only`: The widget becomes visible only when visitor interaction is required and provides the cleanest visitor experience. Most visitors will never see the widget, but suspected bots will encounter the interactive challenge.

Note

Appearance modes only affect visible widget types (Managed and Non-Interactive). Invisible widgets are never shown regardless of the appearance setting.

* Implicit rendering

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>"></div>
  ```

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-appearance="execute"></div>
  ```

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-appearance="interaction-only"></div>
  ```

* Explicit rendering

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>'
    });
  ```

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>',
      appearance: 'execute'
    });
  ```

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>',
      appearance: 'interaction-only'
    });
  ```

***

## Execution modes

Control when the challenge runs and a token is generated.

* `render` (default): The challenge runs automatically after calling the `render()` function and provides immediate protection as soon as the widget loads. The challenge runs in the background while the page loads, ensuring the token is ready when the visitor submits data.

* `execute`: The challenge runs after calling the `turnstile.execute()` function separately and gives you precise control over when verification occurs. This option is useful for multi-step forms, conditional verification, or when you want to defer the challenge until the visitor actually attempts to submit data. This can improve page load performance and visitor experience by only running verification when needed.

  **Common scenarios**

  * Multi-step forms: Run verification only on the final step.
  * Conditional protection: Only verify visitors who meet certain criteria.
  * Performance optimization: Defer verification to reduce initial page load time.
  * User-triggered verification: Let visitors manually start the verification process.

- Implicit rendering

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>"></div>
  ```

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-execution="execute"></div>
  ```

- Explicit rendering

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>'
    });
  ```

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>',
      execution: 'execute'
    });
  ```

  ```js
    turnstile.execute('#widget-container');
  ```

***

## Language configuration

Set the language for the widget interface.

* `auto` (default): Uses the visitor's browser language preference.
* Specific language codes: ISO 639-1 two-letter codes, such as `es`, `fr`, `de`.
* Language and region: Combined codes for regional variants, such as `en-US`, `es-MX`, `pt-BR`.

Notes

* When set to `auto`, Turnstile automatically detects the visitor's preferred language from their browser settings.
* If a requested language is not supported, Turnstile falls back to English.
* Language affects all visitor-facing text including loading messages, error states, and accessibility labels.
* Setting specific languages can improve visitor experience for international audiences.

- Implicit rendering

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>"></div>
  ```

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-language="es"></div>
  ```

  ```html
    <div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-language="en-US"></div>
  ```

- Explicit rendering

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>'
    });
  ```

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>',
      language: 'es'
    });
  ```

***

## Callback configuration

Handle widget events with callbacks.

* `callback`: Triggered when the challenge is successfully completed.
* `error-callback`: Triggered when an error occurs during the challenge.
* `expired-callback`: Triggered when a token expires (before timeout).
* `timeout-callback`: Triggered when an interactive challenge times out.

The success callback receives a token that must be validated on your server using the Siteverify API. Tokens are single-use and expire after 300 seconds (five minutes).

* Implicit rendering

  ```html
    <div class="cf-turnstile"
      data-sitekey="<YOUR-SITE-KEY>"
      data-callback="onSuccess"
      data-error-callback="onError"
      data-expired-callback="onExpired"
      data-timeout-callback="onTimeout"></div>
    <script>
    function onSuccess(token) {
    console.log('Challenge Success:', token);
    }
    function onError(errorCode) {
    console.log('Challenge Error:', errorCode);
    }
    function onExpired() {
    console.log('Token expired');
    }
    function onTimeout() {
    console.log('Challenge timed out');
    }
    </script>
  ```

* Explicit rendering

  ```js
    turnstile.render('#widget-container', {
      sitekey: '<YOUR-SITE-KEY>',
      callback: function(token) {
        console.log('Challenge Success:', token);
      },
      'error-callback': function(errorCode) {
        console.log('Challenge Error:', errorCode);
      },
      'expired-callback': function() {
        console.log('Token expired');
      },
      'timeout-callback': function() {
        console.log('Challenge timed out');
      }
    });
  ```

### Best practices

* Always implement the success callback to handle the token and proceed with form submission or next steps.
* Use error callbacks for graceful error handling and visitor feedback.
* Monitor expired tokens to refresh challenges before they become invalid.
* Handle timeouts to guide visitors through challenge resolution.

***

## Advanced configuration options

### Retry behavior

Control how Turnstile handles failed challenges.

* `auto` (default): Automatically retries failed challenges. Auto retry provides better visitor experience by automatically recovering from temporary network issues or processing errors.
* `never`: Disables automatic retry. This requires manual intervention and gives you full control over error handling in applications that need custom retry logic.
* `retry-interval`: Controls the time between retry attempts (default: 8000ms) and lets you balance between quick recovery and server load.

```html
<div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>"></div>
```

```html
<div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-retry="never"></div>
```

```html
<div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-retry-interval="0000"></div>
```

### Refresh behavior

Control how Turnstile handles token expiration and interactive timeouts.

* `refresh-expired`: Controls behavior when tokens expire (`auto`, `manual`, `never`).
* `refresh-timeout`: Controls behavior when interactive challenges timeout (`auto`, `manual`, `never`).

#### Benefits

* `auto` refresh provides seamless visitor experience but uses more resources.
* `manual` refresh gives visitors control but requires them to take action.
* `never` refresh requires your application to handle all refresh logic.

Different strategies can be used for token expiration versus interactive timeouts based on your visitor experience requirements.

```html
<div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>"></div>
```

```html
<div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-refresh-expired="manual"></div>
```

```html
<div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-refresh-timeout="auto"></div>
```

### Custom data

Add custom identifiers and data to your challenges.

* `action`: A custom identifier for analytics and differentiation (maximum 32 characters).
* `cData`: Custom payload data returned during validation (maximum 255 characters).

#### Use cases

* Action tracking: Differentiate between login, signup, contact forms, and more. in your analytics.
* Visitor context: Pass visitor IDs, session information, or other contextual data.
* A/B testing: Track different widget configurations or page variants.
* Fraud detection: Include additional context for risk assessment.

Warning

Both action and cData fields only accept alphanumeric characters, underscores (\_), and hyphens (-).

```html
<div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-action="login"></div>
```

```html
<div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-cdata="user-cdata"></div>
```

### Form integration

Configure how Turnstile integrates with HTML forms.

When enabled, Turnstile automatically creates a hidden `<input>` element with the verification token. This gets submitted along with your other form data, making server-side validation straightforward.

* `response-field`: Determines whether to create a hidden form field with the token (`default: true`)
* `response-field-name`: Custom name for the hidden form field (`default: cf-turnstile-response`)

#### Benefits

* Automatic form integration means that the token is included when the form is submitted, requiring no additional JavaScript.
* Custom field names helps avoid conflicts with existing form fields.
* Disabled response fields give you full control over token handling for complex form scenarios.

```html
<div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-response-field-name="turnstile-token"></div>
```

```html
<div class="cf-turnstile" data-sitekey="<YOUR-SITE-KEY>" data-response-field="false"></div>
```

***

## Complete configuration reference

| JavaScript Render Parameters | Data Attribute | Description |
| - | - | - |
| `sitekey` | `data-sitekey` | Every widget has a sitekey. This sitekey is associated with the corresponding widget configuration and is created upon the widget creation. |
| `action` | `data-action` | A customer value that can be used to differentiate widgets under the same sitekey in analytics and which is returned upon validation. This can only contain up to 32 alphanumeric characters including `_` and `-`. |
| `cData` | `data-cdata` | A customer payload that can be used to attach customer data to the challenge throughout its issuance and which is returned upon validation. This can only contain up to 255 alphanumeric characters including `_` and `-`. |
| `callback` | `data-callback` | A JavaScript callback invoked upon success of the challenge. The callback is passed a token that can be validated. |
| `error-callback` | `data-error-callback` | A JavaScript callback invoked when there is an error (e.g. network error or the challenge failed). Refer to [Client-side errors](https://developers.cloudflare.com/turnstile/troubleshooting/client-side-errors/). |
| `execution` | `data-execution` | Execution controls when to obtain the token of the widget and can be on `render` (default) or on `execute`. Refer to [Execution Modes](https://developers.cloudflare.com/turnstile/get-started/client-side-rendering/#execution-modes) for more information. |
| `expired-callback` | `data-expired-callback` | A JavaScript callback invoked when the token expires and does not reset the widget. |
| `before-interactive-callback` | `data-before-interactive-callback` | A JavaScript callback invoked before the challenge enters interactive mode. |
| `after-interactive-callback` | `data-after-interactive-callback` | A JavaScript callback invoked when challenge has left interactive mode. |
| `unsupported-callback` | `data-unsupported-callback` | A JavaScript callback invoked when a given client/browser is not supported by Turnstile. |
| `theme` | `data-theme` | The widget theme. Can take the following values: `light`, `dark`, `auto`. The default is `auto`, which respects the visitor preference. This can be forced to light or dark by setting the theme accordingly. |
| `language` | `data-language` | Language to display, must be either: `auto` (default) to use the language that the visitor has chosen, or an ISO 639-1 two-letter language code (e.g. `en`) or language and country code (e.g. `en-US`). Refer to the [list of supported languages](https://developers.cloudflare.com/turnstile/reference/supported-languages/) for more information. |
| `tabindex` | `data-tabindex` | The tabindex of Turnstile's iframe for accessibility purposes. The default value is `0`. |
| `timeout-callback` | `data-timeout-callback` | A JavaScript callback invoked when the challenge presents an interactive challenge but was not solved within a given time. A callback will reset the widget to allow a visitor to solve the challenge again. |
| `response-field` | `data-response-field` | A boolean that controls if an input element with the response token is created, defaults to `true`. |
| `response-field-name` | `data-response-field-name` | Name of the input element, defaults to `cf-turnstile-response`. |
| `size` | `data-size` | The widget size. Can take the following values: `normal`, `flexible`, `compact`. |
| `retry` |  `data-retry` | Controls whether the widget should automatically retry to obtain a token if it did not succeed. The default is `auto`, which will retry automatically. This can be set to `never` to disable retry on failure. |
| `retry-interval` |  `data-retry-interval` | When `retry` is set to `auto`, `retry-interval` controls the time between retry attempts in milliseconds. Value must be a positive integer less than `900000`, defaults to `8000`. |
| `refresh-expired` |  `data-refresh-expired` | Automatically refreshes the token when it expires. Can take `auto`, `manual`, or `never`, defaults to `auto`. |
| `refresh-timeout` | `data-refresh-timeout` | Controls whether the widget should automatically refresh upon entering an interactive challenge and observing a timeout. Can take `auto` (automatically refreshes upon encountering an interactive timeout), `manual` (prompts the visitor to manually refresh) or `never` (will show a timeout), defaults to `auto`. Only applies to widgets of Managed mode. |
| `appearance` | `data-appearance` | Appearance controls when the widget is visible. It can be `always` (default), `execute`, or `interaction-only`. Refer to [Appearance modes](https://developers.cloudflare.com/turnstile/get-started/client-side-rendering/#appearance-modes) for more information. |
| `feedback-enabled` | `data-feedback-enabled` | Allows Cloudflare to gather visitor feedback upon widget failure. It can be `true` (default) or `false`. |

### Examples

```html
<div style="max-width: 500px;">
  <div class="cf-turnstile" data-sitekey=<YOUR-SITE-KEY> data-size="flexible" data-theme="auto"></div>
</div>
```

```html
<div class="cf-turnstile" data-sitekey=<YOUR-SITE-KEY> data-size="compact" data-theme="light" data-language="en">
</div>
```
